/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.models.utils;

import jakarta.transaction.InvalidTransactionException;
import jakarta.transaction.SystemException;
import jakarta.transaction.Transaction;
import java.math.BigInteger;
import java.security.Key;
import java.security.KeyPair;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Base64;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.crypto.spec.SecretKeySpec;
import org.jboss.logging.Logger;
import org.keycloak.Config;
import org.keycloak.broker.social.SocialIdentityProvider;
import org.keycloak.broker.social.SocialIdentityProviderFactory;
import org.keycloak.common.util.CertificateUtils;
import org.keycloak.common.util.KeyUtils;
import org.keycloak.common.util.PemUtils;
import org.keycloak.common.util.SecretGenerator;
import org.keycloak.common.util.Time;
import org.keycloak.component.ComponentModel;
import org.keycloak.deployment.DeployedConfigurationsManager;
import org.keycloak.models.AuthenticationExecutionModel;
import org.keycloak.models.AuthenticationFlowModel;
import org.keycloak.models.AuthenticatorConfigModel;
import org.keycloak.models.ClientModel;
import org.keycloak.models.ClientScopeModel;
import org.keycloak.models.Constants;
import org.keycloak.models.GroupModel;
import org.keycloak.models.GroupProvider;
import org.keycloak.models.GroupProviderFactory;
import org.keycloak.models.IdentityProviderModel;
import org.keycloak.models.KeycloakContext;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.KeycloakSessionFactory;
import org.keycloak.models.KeycloakSessionTask;
import org.keycloak.models.KeycloakSessionTaskWithResult;
import org.keycloak.models.RealmModel;
import org.keycloak.models.RoleModel;
import org.keycloak.models.ScopeContainerModel;
import org.keycloak.models.UserModel;
import org.keycloak.provider.Provider;
import org.keycloak.provider.ProviderFactory;
import org.keycloak.representations.idm.CertificateRepresentation;
import org.keycloak.sessions.AuthenticationSessionModel;
import org.keycloak.sessions.RootAuthenticationSessionModel;
import org.keycloak.transaction.JtaTransactionManagerLookup;
import org.keycloak.transaction.RequestContextHelper;
import org.keycloak.utils.KeycloakSessionUtil;
import org.keycloak.utils.StreamsUtil;

public final class KeycloakModelUtils {
    private static final Logger logger = Logger.getLogger(KeycloakModelUtils.class);
    public static final String AUTH_TYPE_CLIENT_SECRET = "client-secret";
    public static final String AUTH_TYPE_CLIENT_SECRET_JWT = "client-secret-jwt";
    public static final String GROUP_PATH_SEPARATOR = "/";
    public static final String GROUP_PATH_ESCAPE = "~";
    private static final char CLIENT_ROLE_SEPARATOR = '.';
    public static final int MAX_CLIENT_LOOKUPS_DURING_ROLE_RESOLVE = 25;
    public static final int DEFAULT_RSA_KEY_SIZE = 4096;
    public static final int DEFAULT_CERTIFICATE_VALIDITY_YEARS = 3;

    private KeycloakModelUtils() {
    }

    public static String generateId() {
        return UUID.randomUUID().toString();
    }

    public static String generateShortId() {
        return KeycloakModelUtils.generateShortId(UUID.randomUUID());
    }

    public static String generateShortId(UUID uuid) {
        int i;
        byte[] bytes = new byte[16];
        long l = uuid.getMostSignificantBits();
        for (i = 7; i >= 0; --i) {
            bytes[i] = (byte)(l & 0xFFL);
            l >>= 8;
        }
        l = uuid.getLeastSignificantBits();
        for (i = 7; i >= 0; --i) {
            bytes[8 + i] = (byte)(l & 0xFFL);
            l >>= 8;
        }
        return Base64.getUrlEncoder().withoutPadding().encodeToString(bytes);
    }

    public static boolean isValidUUID(String uuid) {
        if (uuid == null) {
            return false;
        }
        try {
            UUID.fromString(uuid);
            return true;
        }
        catch (IllegalArgumentException e) {
            return false;
        }
    }

    public static PublicKey getPublicKey(String publicKeyPem) {
        if (publicKeyPem != null) {
            try {
                return PemUtils.decodePublicKey((String)publicKeyPem);
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
        return null;
    }

    public static X509Certificate getCertificate(String cert) {
        if (cert != null) {
            try {
                return PemUtils.decodeCertificate((String)cert);
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
        return null;
    }

    public static PrivateKey getPrivateKey(String privateKeyPem) {
        if (privateKeyPem != null) {
            try {
                return PemUtils.decodePrivateKey((String)privateKeyPem);
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
        return null;
    }

    public static Key getSecretKey(String secret) {
        return secret != null ? new SecretKeySpec(secret.getBytes(), "HmacSHA256") : null;
    }

    public static String getPemFromKey(Key key) {
        return PemUtils.encodeKey((Key)key);
    }

    public static String getPemFromCertificate(X509Certificate certificate) {
        return PemUtils.encodeCertificate((Certificate)certificate);
    }

    public static CertificateRepresentation generateKeyPairCertificate(String subject) {
        Calendar calendar = Calendar.getInstance();
        calendar.add(1, 3);
        return KeycloakModelUtils.generateKeyPairCertificate(subject, 4096, calendar);
    }

    public static CertificateRepresentation generateKeyPairCertificate(String subject, int keysize, Calendar endDate) {
        KeyPair keyPair = KeyUtils.generateRsaKeyPair((int)keysize);
        X509Certificate certificate = CertificateUtils.generateV1SelfSignedCertificate((KeyPair)keyPair, (String)subject, (BigInteger)BigInteger.valueOf(System.currentTimeMillis()), (Date)endDate.getTime());
        String privateKeyPem = PemUtils.encodeKey((Key)keyPair.getPrivate());
        String certPem = PemUtils.encodeCertificate((Certificate)certificate);
        CertificateRepresentation rep = new CertificateRepresentation();
        rep.setPrivateKey(privateKeyPem);
        rep.setCertificate(certPem);
        return rep;
    }

    public static String generateSecret(ClientModel client) {
        int secretLength = KeycloakModelUtils.getSecretLengthByAuthenticationType(client.getClientAuthenticatorType(), client.getAttribute("token.endpoint.auth.signing.alg"));
        String secret = SecretGenerator.getInstance().randomString(secretLength);
        client.setSecret(secret);
        client.setAttribute("client.secret.creation.time", String.valueOf(Time.currentTime()));
        return secret;
    }

    public static String getDefaultClientAuthenticatorType() {
        return AUTH_TYPE_CLIENT_SECRET;
    }

    public static String generateCodeSecret() {
        return UUID.randomUUID().toString();
    }

    public static ClientModel createManagementClient(RealmModel realm, String name) {
        ClientModel client = KeycloakModelUtils.createClient(realm, name);
        client.setBearerOnly(true);
        return client;
    }

    public static ClientModel createPublicClient(RealmModel realm, String name) {
        ClientModel client = KeycloakModelUtils.createClient(realm, name);
        client.setPublicClient(true);
        return client;
    }

    private static ClientModel createClient(RealmModel realm, String name) {
        ClientModel client = realm.addClient(name);
        client.setClientAuthenticatorType(KeycloakModelUtils.getDefaultClientAuthenticatorType());
        return client;
    }

    public static boolean searchFor(RoleModel role, RoleModel composite, Set<String> visited) {
        if (visited.contains(composite.getId())) {
            return false;
        }
        visited.add(composite.getId());
        if (!composite.isComposite()) {
            return false;
        }
        Set compositeRoles = composite.getCompositesStream().collect(Collectors.toSet());
        return compositeRoles.contains(role) || compositeRoles.stream().anyMatch(x -> x.isComposite() && KeycloakModelUtils.searchFor(role, x, visited));
    }

    public static UserModel findUserByNameOrEmail(KeycloakSession session, RealmModel realm, String username) {
        UserModel user;
        if (realm.isLoginWithEmailAllowed() && username.indexOf(64) != -1 && (user = session.users().getUserByEmail(realm, username)) != null) {
            return user;
        }
        return session.users().getUserByUsername(realm, username);
    }

    public static void runJobInTransaction(KeycloakSessionFactory factory, KeycloakSessionTask task) {
        KeycloakModelUtils.runJobInTransaction(factory, null, task);
    }

    public static void runJobInTransaction(KeycloakSessionFactory factory, KeycloakContext context, KeycloakSessionTask task) {
        KeycloakModelUtils.runJobInTransactionWithResult(factory, context, session -> {
            task.run(session);
            return null;
        }, task.getTaskName());
    }

    public static void cloneContextRealmClientToSession(KeycloakContext origContext, KeycloakSession targetSession) {
        KeycloakModelUtils.cloneContextToSession(origContext, targetSession, false);
    }

    public static void cloneContextRealmClientSessionToSession(KeycloakContext origContext, KeycloakSession targetSession) {
        KeycloakModelUtils.cloneContextToSession(origContext, targetSession, true);
    }

    private static void cloneContextToSession(KeycloakContext origContext, KeycloakSession targetSession, boolean includeAuthenticatedSessionModel) {
        if (origContext == null) {
            return;
        }
        RealmModel realmModel = null;
        if (origContext.getRealm() != null && (realmModel = targetSession.realms().getRealm(origContext.getRealm().getId())) != null) {
            targetSession.getContext().setRealm(realmModel);
        }
        ClientModel clientModel = null;
        if (origContext.getClient() != null) {
            if (origContext.getRealm() == null || !Objects.equals(origContext.getRealm().getId(), origContext.getClient().getRealm().getId())) {
                realmModel = targetSession.realms().getRealm(origContext.getClient().getRealm().getId());
            }
            if (realmModel != null && (clientModel = targetSession.clients().getClientById(realmModel, origContext.getClient().getId())) != null) {
                targetSession.getContext().setClient(clientModel);
            }
        }
        if (includeAuthenticatedSessionModel && origContext.getAuthenticationSession() != null) {
            AuthenticationSessionModel authSessionModel;
            RootAuthenticationSessionModel rootAuthSession;
            if (origContext.getClient() == null || !Objects.equals(origContext.getClient().getId(), origContext.getAuthenticationSession().getClient().getId())) {
                realmModel = origContext.getRealm() == null || !Objects.equals(origContext.getRealm().getId(), origContext.getAuthenticationSession().getRealm().getId()) ? targetSession.realms().getRealm(origContext.getAuthenticationSession().getRealm().getId()) : targetSession.getContext().getRealm();
                ClientModel clientModel2 = clientModel = realmModel != null ? targetSession.clients().getClientById(realmModel, origContext.getAuthenticationSession().getClient().getId()) : null;
            }
            if (clientModel != null && (rootAuthSession = targetSession.authenticationSessions().getRootAuthenticationSession(realmModel, origContext.getAuthenticationSession().getParentSession().getId())) != null && (authSessionModel = rootAuthSession.getAuthenticationSession(clientModel, origContext.getAuthenticationSession().getTabId())) != null) {
                targetSession.getContext().setAuthenticationSession(authSessionModel);
            }
        }
    }

    public static <V> V runJobInTransactionWithResult(KeycloakSessionFactory factory, KeycloakSessionTaskWithResult<V> callable) {
        return KeycloakModelUtils.runJobInTransactionWithResult(factory, null, callable, "Non-HTTP task");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static <V> V runJobInTransactionWithResult(KeycloakSessionFactory factory, KeycloakContext context, KeycloakSessionTaskWithResult<V> callable, String taskName) {
        Object result;
        KeycloakSession existing = KeycloakSessionUtil.getKeycloakSession();
        try (KeycloakSession session = factory.create();){
            RequestContextHelper.getContext(session).setContextMessage(taskName);
            session.getTransactionManager().begin();
            KeycloakSessionUtil.setKeycloakSession((KeycloakSession)session);
            try {
                KeycloakModelUtils.cloneContextRealmClientToSession(context, session);
                result = callable.run(session);
            }
            catch (Throwable t) {
                session.getTransactionManager().setRollbackOnly();
                throw t;
            }
        }
        finally {
            KeycloakSessionUtil.setKeycloakSession((KeycloakSession)existing);
        }
        return (V)result;
    }

    public static void runJobInTransactionWithTimeout(KeycloakSessionFactory factory, KeycloakSessionTask task, int timeoutInSeconds) {
        try {
            KeycloakModelUtils.setTransactionLimit(factory, timeoutInSeconds);
            KeycloakModelUtils.runJobInTransaction(factory, task);
        }
        finally {
            KeycloakModelUtils.setTransactionLimit(factory, 0);
        }
    }

    public static void setTransactionLimit(KeycloakSessionFactory factory, int timeoutInSeconds) {
        JtaTransactionManagerLookup lookup = (JtaTransactionManagerLookup)factory.getProviderFactory(JtaTransactionManagerLookup.class);
        if (lookup != null && lookup.getTransactionManager() != null) {
            try {
                lookup.getTransactionManager().setTransactionTimeout(timeoutInSeconds);
            }
            catch (SystemException e) {
                throw new RuntimeException(e);
            }
        }
    }

    public static Function<KeycloakSessionFactory, ComponentModel> componentModelGetter(String realmId, String componentId) {
        return factory -> KeycloakModelUtils.getComponentModel(factory, realmId, componentId);
    }

    public static ComponentModel getComponentModel(KeycloakSessionFactory factory, String realmId, String componentId) {
        AtomicReference cm = new AtomicReference();
        KeycloakModelUtils.runJobInTransaction(factory, session -> {
            RealmModel realm = session.realms().getRealm(realmId);
            cm.set(realm == null ? null : realm.getComponent(componentId));
        });
        return (ComponentModel)cm.get();
    }

    public static <T extends Provider> ProviderFactory<T> getComponentFactory(KeycloakSessionFactory factory, Class<T> providerClass, Config.Scope config, String spiName) {
        String realmId = config.get("realmId");
        String componentId = config.get("componentId");
        if (realmId == null || componentId == null) {
            realmId = "ROOT";
            ScopeComponentModel cm = new ScopeComponentModel(providerClass, config, spiName, realmId);
            return factory.getProviderFactory(providerClass, realmId, cm.getId(), k -> cm);
        }
        return factory.getProviderFactory(providerClass, realmId, componentId, KeycloakModelUtils.componentModelGetter(realmId, componentId));
    }

    public static String getMasterRealmAdminManagementClientId(String realmName) {
        return realmName + "-realm";
    }

    public static ComponentModel createComponentModel(String name, String parentId, String providerId, String providerType, String ... config) {
        ComponentModel mapperModel = new ComponentModel();
        mapperModel.setParentId(parentId);
        mapperModel.setName(name);
        mapperModel.setProviderId(providerId);
        mapperModel.setProviderType(providerType);
        String key = null;
        for (String configEntry : config) {
            if (key == null) {
                key = configEntry;
                continue;
            }
            mapperModel.getConfig().add((Object)key, (Object)configEntry);
            key = null;
        }
        if (key != null) {
            throw new IllegalStateException("Invalid count of arguments for config. Maybe mistake?");
        }
        return mapperModel;
    }

    public static String toLowerCaseSafe(String str) {
        return str == null ? null : str.toLowerCase();
    }

    public static void setupDefaultRole(RealmModel realm, String defaultRoleName) {
        RoleModel defaultRole = realm.addRole(defaultRoleName);
        defaultRole.setDescription("${role_default-roles}");
        realm.setDefaultRole(defaultRole);
    }

    public static RoleModel setupOfflineRole(RealmModel realm) {
        RoleModel offlineRole = realm.getRole("offline_access");
        if (offlineRole == null) {
            offlineRole = realm.addRole("offline_access");
            offlineRole.setDescription("${role_offline-access}");
            realm.addToDefaultRoles(offlineRole);
        }
        return offlineRole;
    }

    public static void setupDeleteAccount(ClientModel accountClient) {
        RoleModel deleteOwnAccount = accountClient.getRole("delete-account");
        if (deleteOwnAccount == null) {
            deleteOwnAccount = accountClient.addRole("delete-account");
        }
        deleteOwnAccount.setDescription("${role_delete-account}");
    }

    public static void deepFindAuthenticationExecutions(RealmModel realm, AuthenticationFlowModel flow, List<AuthenticationExecutionModel> result) {
        realm.getAuthenticationExecutionsStream(flow.getId()).forEachOrdered(execution -> {
            if (execution.isAuthenticatorFlow()) {
                AuthenticationFlowModel subFlow = realm.getAuthenticationFlowById(execution.getFlowId());
                KeycloakModelUtils.deepFindAuthenticationExecutions(realm, subFlow, result);
            } else {
                result.add((AuthenticationExecutionModel)execution);
            }
        });
    }

    public static Collection<String> resolveAttribute(GroupModel group, String name, boolean aggregateAttrs) {
        Set<String> values = group.getAttributeStream(name).collect(Collectors.toSet());
        if ((values.isEmpty() || aggregateAttrs) && group.getParentId() != null) {
            values.addAll(KeycloakModelUtils.resolveAttribute(group.getParent(), name, aggregateAttrs));
        }
        return values;
    }

    public static Collection<String> resolveAttribute(UserModel user, String name, boolean aggregateAttrs) {
        List<String> values = user.getAttributeStream(name).collect(Collectors.toList());
        HashSet<String> aggrValues = new HashSet<String>();
        if (!values.isEmpty()) {
            if (!aggregateAttrs) {
                return values;
            }
            aggrValues.addAll(values);
        }
        Stream<Collection> attributes = user.getGroupsStream().map(group -> KeycloakModelUtils.resolveAttribute(group, name, aggregateAttrs)).filter(attr -> !attr.isEmpty());
        if (!aggregateAttrs) {
            Optional<Collection> first = attributes.findFirst();
            if (first.isPresent()) {
                return first.get();
            }
        } else {
            aggrValues.addAll(attributes.flatMap(Collection::stream).collect(Collectors.toSet()));
        }
        return aggrValues;
    }

    private static GroupModel findSubGroup(String[] segments, int index, GroupModel parent) {
        return parent.getSubGroupsStream().map(group -> {
            String[] pathSegments;
            String groupName = group.getName();
            if (groupName.equals((pathSegments = KeycloakModelUtils.formatPathSegments(segments, index, groupName))[index])) {
                GroupModel found;
                if (pathSegments.length == index + 1) {
                    return group;
                }
                if (index + 1 < pathSegments.length && (found = KeycloakModelUtils.findSubGroup(pathSegments, index + 1, group)) != null) {
                    return found;
                }
            }
            return null;
        }).filter(Objects::nonNull).findFirst().orElse(null);
    }

    private static String[] formatPathSegments(String[] segments, int index, String groupName) {
        String[] nameSegments = groupName.split(GROUP_PATH_SEPARATOR);
        if (nameSegments.length > 1 && segments.length >= nameSegments.length) {
            for (int i = 0; i < nameSegments.length; ++i) {
                if (nameSegments[i].equals(segments[index + i])) continue;
                return segments;
            }
            int numMergedIndexes = nameSegments.length - 1;
            String[] newPath = new String[segments.length - numMergedIndexes];
            for (int i = 0; i < newPath.length; ++i) {
                newPath[i] = i == index ? groupName : (i > index ? segments[i + numMergedIndexes] : segments[i]);
            }
            return newPath;
        }
        return segments;
    }

    public static boolean escapeSlashesInGroupPath(KeycloakSession session) {
        GroupProviderFactory fact = (GroupProviderFactory)session.getKeycloakSessionFactory().getProviderFactory(GroupProvider.class);
        return fact.escapeSlashesInGroupPath();
    }

    public static GroupModel findGroupByPath(KeycloakSession session, RealmModel realm, String path) {
        if (path == null) {
            return null;
        }
        String[] split = KeycloakModelUtils.splitPath(path, KeycloakModelUtils.escapeSlashesInGroupPath(session));
        if (split.length == 0) {
            return null;
        }
        return KeycloakModelUtils.getGroupModel(session.groups(), realm, null, split, 0);
    }

    public static GroupModel findGroupByPath(KeycloakSession session, RealmModel realm, String[] path) {
        if (path == null || path.length == 0) {
            return null;
        }
        return KeycloakModelUtils.getGroupModel(session.groups(), realm, null, path, 0);
    }

    private static GroupModel getGroupModel(GroupProvider groupProvider, RealmModel realm, GroupModel parent, String[] split, int index) {
        StringBuilder nameBuilder = new StringBuilder();
        for (int i = index; i < split.length; ++i) {
            nameBuilder.append(split[i]);
            GroupModel group = groupProvider.getGroupByName(realm, parent, nameBuilder.toString());
            if (group != null) {
                if (i < split.length - 1) {
                    return KeycloakModelUtils.getGroupModel(groupProvider, realm, group, split, i + 1);
                }
                return group;
            }
            nameBuilder.append(GROUP_PATH_SEPARATOR);
        }
        return null;
    }

    public static String[] splitPath(String path, boolean escapedSlashes) {
        if (path == null) {
            return null;
        }
        if (path.startsWith(GROUP_PATH_SEPARATOR)) {
            path = path.substring(1);
        }
        if (path.endsWith(GROUP_PATH_SEPARATOR)) {
            path = path.substring(0, path.length() - 1);
        }
        return escapedSlashes ? (String[])Arrays.stream(path.split("(?<!" + Pattern.quote(GROUP_PATH_ESCAPE) + ")" + Pattern.quote(GROUP_PATH_SEPARATOR))).map(KeycloakModelUtils::unescapeGroupNameForPath).toArray(String[]::new) : path.split(GROUP_PATH_SEPARATOR);
    }

    private static String escapeGroupNameForPath(String groupName) {
        return groupName.replace(GROUP_PATH_SEPARATOR, "~/");
    }

    private static String unescapeGroupNameForPath(String groupName) {
        return groupName.replace("~/", GROUP_PATH_SEPARATOR);
    }

    public static String buildGroupPath(boolean escapeSlashes, String ... names) {
        StringBuilder sb = new StringBuilder();
        sb.append(GROUP_PATH_SEPARATOR);
        for (int i = 0; i < names.length; ++i) {
            sb.append(escapeSlashes ? KeycloakModelUtils.escapeGroupNameForPath(names[i]) : names[i]);
            if (i >= names.length - 1) continue;
            sb.append(GROUP_PATH_SEPARATOR);
        }
        return sb.toString();
    }

    private static void buildGroupPath(StringBuilder sb, String groupName, GroupModel parent, boolean escapeSlashes) {
        if (parent != null) {
            KeycloakModelUtils.buildGroupPath(sb, parent.getName(), parent.getParent(), escapeSlashes);
        }
        sb.append(GROUP_PATH_SEPARATOR).append(escapeSlashes ? KeycloakModelUtils.escapeGroupNameForPath(groupName) : groupName);
    }

    public static String buildGroupPath(GroupModel group) {
        StringBuilder sb = new StringBuilder();
        KeycloakModelUtils.buildGroupPath(sb, group.getName(), group.getParent(), group.escapeSlashesInGroupPath());
        return sb.toString();
    }

    public static String buildGroupPath(GroupModel group, GroupModel otherParentGroup) {
        StringBuilder sb = new StringBuilder();
        KeycloakModelUtils.buildGroupPath(sb, group.getName(), otherParentGroup, group.escapeSlashesInGroupPath());
        return sb.toString();
    }

    public static String normalizeGroupPath(String groupPath) {
        if (groupPath == null) {
            return null;
        }
        Object normalized = groupPath;
        if (!((String)normalized).startsWith(GROUP_PATH_SEPARATOR)) {
            normalized = GROUP_PATH_SEPARATOR + (String)normalized;
        }
        if (((String)normalized).endsWith(GROUP_PATH_SEPARATOR)) {
            normalized = ((String)normalized).substring(0, ((String)normalized).length() - 1);
        }
        return normalized;
    }

    public static Stream<RoleModel> getClientScopeMappingsStream(ClientModel client, ScopeContainerModel container) {
        return container.getScopeMappingsStream().filter(role -> role.getContainer() instanceof ClientModel && Objects.equals(client.getId(), role.getContainer().getId()));
    }

    public static RoleModel getRoleFromString(RealmModel realm, String roleName) {
        int counter;
        if (roleName == null) {
            return null;
        }
        int scopeIndex = roleName.lastIndexOf(46);
        for (counter = 0; scopeIndex >= 0 && counter < 25; ++counter) {
            String appName = roleName.substring(0, scopeIndex);
            ClientModel client = realm.getClientByClientId(appName);
            if (client != null) {
                String role = roleName.substring(scopeIndex + 1);
                return client.getRole(role);
            }
            scopeIndex = roleName.lastIndexOf(46, scopeIndex - 1);
        }
        if (counter >= 25) {
            logger.warnf("Not able to retrieve role model from the role name '%s'. Please use shorter role names with the limited amount of dots, roleName", roleName.length() > 100 ? roleName.substring(0, 100) + "..." : roleName);
            return null;
        }
        return realm.getRole(roleName);
    }

    public static String[] parseRole(String role) {
        int scopeIndex = role.lastIndexOf(46);
        if (scopeIndex > -1) {
            String appName = role.substring(0, scopeIndex);
            role = role.substring(scopeIndex + 1);
            String[] rtn = new String[]{appName, role};
            return rtn;
        }
        String[] rtn = new String[]{null, role};
        return rtn;
    }

    public static String buildRoleQualifier(String clientId, String roleName) {
        if (clientId == null) {
            return roleName;
        }
        return clientId + "." + roleName;
    }

    public static boolean isFlowUsed(KeycloakSession session, RealmModel realm, AuthenticationFlowModel model) {
        Stream directGrantFlowOverridingClients;
        AuthenticationFlowModel realmFlow = null;
        realmFlow = realm.getBrowserFlow();
        if (realmFlow != null && realmFlow.getId().equals(model.getId())) {
            return true;
        }
        realmFlow = realm.getRegistrationFlow();
        if (realmFlow != null && realmFlow.getId().equals(model.getId())) {
            return true;
        }
        realmFlow = realm.getClientAuthenticationFlow();
        if (realmFlow != null && realmFlow.getId().equals(model.getId())) {
            return true;
        }
        realmFlow = realm.getDirectGrantFlow();
        if (realmFlow != null && realmFlow.getId().equals(model.getId())) {
            return true;
        }
        realmFlow = realm.getResetCredentialsFlow();
        if (realmFlow != null && realmFlow.getId().equals(model.getId())) {
            return true;
        }
        realmFlow = realm.getDockerAuthenticationFlow();
        if (realmFlow != null && realmFlow.getId().equals(model.getId())) {
            return true;
        }
        realmFlow = realm.getFirstBrokerLoginFlow();
        if (realmFlow != null && realmFlow.getId().equals(model.getId())) {
            return true;
        }
        Stream browserFlowOverridingClients = realm.searchClientByAuthenticationFlowBindingOverrides(Collections.singletonMap("browser", model.getId()), Integer.valueOf(0), Integer.valueOf(1));
        boolean usedByClient = StreamsUtil.closing(Stream.concat(browserFlowOverridingClients, directGrantFlowOverridingClients = realm.searchClientByAuthenticationFlowBindingOverrides(Collections.singletonMap("direct_grant", model.getId()), Integer.valueOf(0), Integer.valueOf(1)))).limit(1L).findAny().isPresent();
        if (usedByClient) {
            return true;
        }
        return session.identityProviders().getByFlow(model.getId(), null, Integer.valueOf(0), Integer.valueOf(1)).findAny().isPresent();
    }

    public static void deepDeleteAuthenticationFlow(KeycloakSession session, RealmModel realm, AuthenticationFlowModel authFlow, Runnable flowUnavailableHandler, Runnable builtinFlowHandler) {
        if (authFlow == null) {
            flowUnavailableHandler.run();
            return;
        }
        if (authFlow.isBuiltIn()) {
            builtinFlowHandler.run();
        }
        realm.getAuthenticationExecutionsStream(authFlow.getId()).forEachOrdered(authExecutor -> KeycloakModelUtils.deepDeleteAuthenticationExecutor(session, realm, authExecutor, flowUnavailableHandler, builtinFlowHandler));
        realm.removeAuthenticationFlow(authFlow);
    }

    public static void deepDeleteAuthenticationExecutor(KeycloakSession session, RealmModel realm, AuthenticationExecutionModel authExecutor, Runnable flowUnavailableHandler, Runnable builtinFlowHandler) {
        AuthenticatorConfigModel config;
        DeployedConfigurationsManager configManager;
        if (authExecutor == null) {
            flowUnavailableHandler.run();
            return;
        }
        if (authExecutor.getFlowId() != null) {
            AuthenticationFlowModel authFlow = realm.getAuthenticationFlowById(authExecutor.getFlowId());
            KeycloakModelUtils.deepDeleteAuthenticationFlow(session, realm, authFlow, flowUnavailableHandler, builtinFlowHandler);
        }
        if (authExecutor.getAuthenticatorConfig() != null && (configManager = new DeployedConfigurationsManager(session)).getDeployedAuthenticatorConfig(authExecutor.getAuthenticatorConfig()) == null && (config = configManager.getAuthenticatorConfig(realm, authExecutor.getAuthenticatorConfig())) != null) {
            realm.removeAuthenticatorConfig(config);
        }
        realm.removeAuthenticatorExecution(authExecutor);
    }

    public static ClientScopeModel getClientScopeByName(RealmModel realm, String clientScopeName) {
        return realm.getClientScopesStream().filter(clientScope -> Objects.equals(clientScopeName, clientScope.getName())).findFirst().orElseGet(() -> realm.getClientByClientId(clientScopeName));
    }

    public static ClientScopeModel findClientScopeById(RealmModel realm, ClientModel client, String clientScopeId) {
        if (client.getId().equals(clientScopeId)) {
            return client;
        }
        ClientScopeModel clientScope = realm.getClientScopeById(clientScopeId);
        if (clientScope == null) {
            clientScope = client.getDynamicClientScope(clientScopeId);
        }
        if (clientScope != null) {
            return clientScope;
        }
        return realm.getClientById(clientScopeId);
    }

    public static String convertClientScopeName(String previousName) {
        if (previousName.contains(" ")) {
            return previousName.replaceAll(" ", "_");
        }
        return previousName;
    }

    public static void setupAuthorizationServices(RealmModel realm) {
        for (String roleName : Constants.AUTHZ_DEFAULT_AUTHORIZATION_ROLES) {
            if (realm.getRole(roleName) != null) continue;
            RoleModel role = realm.addRole(roleName);
            role.setDescription("${role_" + roleName + "}");
            realm.addToDefaultRoles(role);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static void suspendJtaTransaction(KeycloakSessionFactory factory, Runnable runnable) {
        JtaTransactionManagerLookup lookup = (JtaTransactionManagerLookup)factory.getProviderFactory(JtaTransactionManagerLookup.class);
        Transaction suspended = null;
        try {
            if (lookup != null && lookup.getTransactionManager() != null) {
                try {
                    suspended = lookup.getTransactionManager().suspend();
                }
                catch (SystemException e) {
                    throw new RuntimeException(e);
                }
            }
            runnable.run();
            if (suspended == null) return;
        }
        catch (Throwable throwable) {
            if (suspended == null) throw throwable;
            try {
                lookup.getTransactionManager().resume(suspended);
                throw throwable;
            }
            catch (InvalidTransactionException | SystemException e) {
                throw new RuntimeException(e);
            }
        }
        try {
            lookup.getTransactionManager().resume(suspended);
            return;
        }
        catch (InvalidTransactionException | SystemException e) {
            throw new RuntimeException(e);
        }
    }

    public static String getIdentityProviderDisplayName(KeycloakSession session, IdentityProviderModel provider) {
        String displayName = provider.getDisplayName();
        if (displayName != null && !displayName.isEmpty()) {
            return displayName;
        }
        SocialIdentityProviderFactory providerFactory = (SocialIdentityProviderFactory)session.getKeycloakSessionFactory().getProviderFactory(SocialIdentityProvider.class, provider.getProviderId());
        if (providerFactory != null) {
            return providerFactory.getName();
        }
        return provider.getAlias();
    }

    public static int getSecretLengthByAuthenticationType(String clientAuthenticatorType, String signingAlg) {
        if (clientAuthenticatorType != null) {
            switch (clientAuthenticatorType) {
                case "client-secret-jwt": {
                    if ("HS384".equals(signingAlg)) {
                        return SecretGenerator.equivalentEntropySize((int)48, (int)SecretGenerator.ALPHANUM.length);
                    }
                    if ("HS512".equals(signingAlg)) {
                        return SecretGenerator.equivalentEntropySize((int)64, (int)SecretGenerator.ALPHANUM.length);
                    }
                    return SecretGenerator.equivalentEntropySize((int)32, (int)SecretGenerator.ALPHANUM.length);
                }
            }
        }
        return 32;
    }

    public static void setDefaultGroups(KeycloakSession session, RealmModel realm, Stream<String> groups) {
        realm.getDefaultGroupsStream().collect(Collectors.toList()).forEach(arg_0 -> ((RealmModel)realm).removeDefaultGroup(arg_0));
        groups.forEach(path -> {
            GroupModel found = KeycloakModelUtils.findGroupByPath(session, realm, path);
            if (found == null) {
                throw new RuntimeException("default group in realm rep doesn't exist: " + path);
            }
            realm.addDefaultGroup(found);
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static <T> T runOnRealm(KeycloakSession session, RealmModel target, Function<KeycloakSession, T> operation) {
        KeycloakContext context = session.getContext();
        RealmModel currentRealm = context.getRealm();
        if (currentRealm.equals((Object)target)) {
            return operation.apply(session);
        }
        try {
            context.setRealm(target);
            T t = operation.apply(session);
            return t;
        }
        finally {
            context.setRealm(currentRealm);
        }
    }

    public static List<String> getAcceptedClientScopeProtocols(ClientModel client) {
        List<String> acceptedClientProtocols = client.getProtocol() == null || "openid-connect".equals(client.getProtocol()) ? List.of("openid-connect", "oid4vc") : List.of(client.getProtocol());
        return acceptedClientProtocols;
    }

    private static class ScopeComponentModel
    extends ComponentModel {
        private final String componentId;
        private final String providerId;
        private final String providerType;
        private final String realmId;
        private final Config.Scope config;

        public ScopeComponentModel(Class<?> providerClass, Config.Scope baseConfiguration, String spiName, String realmId) {
            String pr = baseConfiguration.get("provider", Config.getProvider((String)spiName));
            this.providerId = pr == null ? "default" : pr;
            this.config = baseConfiguration.scope(new String[]{this.providerId});
            this.componentId = spiName + "- " + (String)(realmId == null ? "" : "f:" + realmId + ":") + this.providerId;
            this.realmId = realmId;
            this.providerType = providerClass.getName();
        }

        public String getProviderType() {
            return this.providerType;
        }

        public String getProviderId() {
            return this.providerId;
        }

        public String getName() {
            return this.componentId + "-config";
        }

        public String getId() {
            return this.componentId;
        }

        public String getParentId() {
            return this.realmId;
        }

        public boolean get(String key, boolean defaultValue) {
            return this.config.getBoolean(key, Boolean.valueOf(defaultValue));
        }

        public long get(String key, long defaultValue) {
            return this.config.getLong(key, Long.valueOf(defaultValue));
        }

        public int get(String key, int defaultValue) {
            return this.config.getInt(key, Integer.valueOf(defaultValue));
        }

        public String get(String key, String defaultValue) {
            return this.config.get(key, defaultValue);
        }

        public String get(String key) {
            return this.get(key, null);
        }
    }
}

