﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/managedblockchain/ManagedBlockchain_EXPORTS.h>
#include <aws/managedblockchain/model/InvitationStatus.h>
#include <aws/managedblockchain/model/NetworkSummary.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace ManagedBlockchain {
namespace Model {

/**
 * <p>An invitation to an Amazon Web Services account to create a member and join
 * the network.</p> <p>Applies only to Hyperledger Fabric.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/Invitation">AWS
 * API Reference</a></p>
 */
class Invitation {
 public:
  AWS_MANAGEDBLOCKCHAIN_API Invitation() = default;
  AWS_MANAGEDBLOCKCHAIN_API Invitation(Aws::Utils::Json::JsonView jsonValue);
  AWS_MANAGEDBLOCKCHAIN_API Invitation& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_MANAGEDBLOCKCHAIN_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The unique identifier for the invitation.</p>
   */
  inline const Aws::String& GetInvitationId() const { return m_invitationId; }
  inline bool InvitationIdHasBeenSet() const { return m_invitationIdHasBeenSet; }
  template <typename InvitationIdT = Aws::String>
  void SetInvitationId(InvitationIdT&& value) {
    m_invitationIdHasBeenSet = true;
    m_invitationId = std::forward<InvitationIdT>(value);
  }
  template <typename InvitationIdT = Aws::String>
  Invitation& WithInvitationId(InvitationIdT&& value) {
    SetInvitationId(std::forward<InvitationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time that the invitation was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationDate() const { return m_creationDate; }
  inline bool CreationDateHasBeenSet() const { return m_creationDateHasBeenSet; }
  template <typename CreationDateT = Aws::Utils::DateTime>
  void SetCreationDate(CreationDateT&& value) {
    m_creationDateHasBeenSet = true;
    m_creationDate = std::forward<CreationDateT>(value);
  }
  template <typename CreationDateT = Aws::Utils::DateTime>
  Invitation& WithCreationDate(CreationDateT&& value) {
    SetCreationDate(std::forward<CreationDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time that the invitation expires. This is the
   * <code>CreationDate</code> plus the <code>ProposalDurationInHours</code> that is
   * specified in the <code>ProposalThresholdPolicy</code>. After this date and time,
   * the invitee can no longer create a member and join the network using this
   * <code>InvitationId</code>.</p>
   */
  inline const Aws::Utils::DateTime& GetExpirationDate() const { return m_expirationDate; }
  inline bool ExpirationDateHasBeenSet() const { return m_expirationDateHasBeenSet; }
  template <typename ExpirationDateT = Aws::Utils::DateTime>
  void SetExpirationDate(ExpirationDateT&& value) {
    m_expirationDateHasBeenSet = true;
    m_expirationDate = std::forward<ExpirationDateT>(value);
  }
  template <typename ExpirationDateT = Aws::Utils::DateTime>
  Invitation& WithExpirationDate(ExpirationDateT&& value) {
    SetExpirationDate(std::forward<ExpirationDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the invitation:</p> <ul> <li> <p> <code>PENDING</code> - The
   * invitee hasn't created a member to join the network, and the invitation hasn't
   * yet expired.</p> </li> <li> <p> <code>ACCEPTING</code> - The invitee has begun
   * creating a member, and creation hasn't yet completed.</p> </li> <li> <p>
   * <code>ACCEPTED</code> - The invitee created a member and joined the network
   * using the <code>InvitationID</code>.</p> </li> <li> <p> <code>REJECTED</code> -
   * The invitee rejected the invitation.</p> </li> <li> <p> <code>EXPIRED</code> -
   * The invitee neither created a member nor rejected the invitation before the
   * <code>ExpirationDate</code>.</p> </li> </ul>
   */
  inline InvitationStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(InvitationStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline Invitation& WithStatus(InvitationStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{

  inline const NetworkSummary& GetNetworkSummary() const { return m_networkSummary; }
  inline bool NetworkSummaryHasBeenSet() const { return m_networkSummaryHasBeenSet; }
  template <typename NetworkSummaryT = NetworkSummary>
  void SetNetworkSummary(NetworkSummaryT&& value) {
    m_networkSummaryHasBeenSet = true;
    m_networkSummary = std::forward<NetworkSummaryT>(value);
  }
  template <typename NetworkSummaryT = NetworkSummary>
  Invitation& WithNetworkSummary(NetworkSummaryT&& value) {
    SetNetworkSummary(std::forward<NetworkSummaryT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the invitation. For more information about
   * ARNs and their format, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Names (ARNs)</a> in the <i>Amazon Web Services General
   * Reference</i>.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  Invitation& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_invitationId;
  bool m_invitationIdHasBeenSet = false;

  Aws::Utils::DateTime m_creationDate{};
  bool m_creationDateHasBeenSet = false;

  Aws::Utils::DateTime m_expirationDate{};
  bool m_expirationDateHasBeenSet = false;

  InvitationStatus m_status{InvitationStatus::NOT_SET};
  bool m_statusHasBeenSet = false;

  NetworkSummary m_networkSummary;
  bool m_networkSummaryHasBeenSet = false;

  Aws::String m_arn;
  bool m_arnHasBeenSet = false;
};

}  // namespace Model
}  // namespace ManagedBlockchain
}  // namespace Aws
