# frozen_string_literal: true

require 'gitlab/housekeeper/push_options'

module Gitlab
  module Housekeeper
    class Change
      attr_accessor :identifiers,
        :title,
        :description,
        :changed_files,
        :labels,
        :keep_class,
        :changelog_type,
        :changelog_ee,
        :mr_web_url,
        :push_options,
        :non_housekeeper_changes,
        :context
      attr_reader :assignees,
        :reviewers

      def initialize
        @labels = []
        @assignees = []
        @reviewers = []
        @non_housekeeper_changes = []
        @push_options = PushOptions.new
        @aborted = false
      end

      def assignees=(assignees)
        @assignees = Array(assignees)
      end

      def reviewers=(reviewers)
        @reviewers = Array(reviewers)
      end

      def abort!
        @aborted = true
      end

      def aborted?
        @aborted
      end

      def mr_description
        <<~MARKDOWN
        #{description}

        This change was generated by
        [gitlab-housekeeper](https://gitlab.com/gitlab-org/gitlab/-/tree/master/gems/gitlab-housekeeper)
        #{generated_by} using the `#{keep_class}` keep.

        To provide feedback on your experience with `gitlab-housekeeper` please create an issue with the
        label ~"GitLab Housekeeper" and consider pinging the author of this keep.
        MARKDOWN
      end

      def commit_message
        <<~MARKDOWN.chomp
        #{truncated_title}

        #{mr_description}

        Changelog: #{changelog_type || 'other'}
        #{changelog_ee ? "EE: true\n" : ''}
        MARKDOWN
      end

      def update_required?(category)
        !category.in?(non_housekeeper_changes)
      end

      def already_approved?
        non_housekeeper_changes.include?(:approvals)
      end

      def valid?
        @identifiers && @title && @description && @changed_files.present?
      end

      def truncated_title
        truncate(title, 72)
      end

      def truncate(text, limit)
        return text if text.length <= limit

        text[0...limit]
      end

      private

      def generated_by
        job_url = ENV['CI_JOB_URL']

        if job_url
          "in [CI](#{job_url})"
        else
          'locally'
        end
      end
    end
  end
end
